<?php 
require_once 'config.php';
session_start();

// Check if user is logged in, if not redirect to login
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'] ?? 'User';

// Handle Profile Updates
$profile_message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'update_profile') {
        $name = isset($_POST['name']) ? trim($_POST['name']) : '';
        $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
        
        if (!empty($name) && !empty($phone)) {
            $update_query = "UPDATE users SET name = ?, phone = ? WHERE id = ?";
            $stmt = $conn->prepare($update_query);
            if ($stmt) {
                $stmt->bind_param('ssi', $name, $phone, $user_id);
                if ($stmt->execute()) {
                    $_SESSION['user_name'] = $name;
                    $profile_message = '<div class="alert alert-success">Profile updated successfully!</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error updating profile.</div>';
                }
                $stmt->close();
            }
        }
    } elseif ($_POST['action'] === 'update_password') {
        $old_password = isset($_POST['old_password']) ? $_POST['old_password'] : '';
        $new_password = isset($_POST['new_password']) ? $_POST['new_password'] : '';
        $confirm_password = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';
        
        // Verify old password
        $user_query = "SELECT password FROM users WHERE id = ?";
        $stmt = $conn->prepare($user_query);
        if ($stmt) {
            $stmt->bind_param('i', $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            if ($user && password_verify($old_password, $user['password'])) {
                if ($new_password === $confirm_password && strlen($new_password) >= 6) {
                    $hashed = password_hash($new_password, PASSWORD_DEFAULT);
                    $pwd_query = "UPDATE users SET password = ? WHERE id = ?";
                    $pwd_stmt = $conn->prepare($pwd_query);
                    if ($pwd_stmt) {
                        $pwd_stmt->bind_param('si', $hashed, $user_id);
                        if ($pwd_stmt->execute()) {
                            $profile_message = '<div class="alert alert-success">Password changed successfully!</div>';
                        }
                        $pwd_stmt->close();
                    }
                } else {
                    $profile_message = '<div class="alert alert-danger">Passwords do not match or are too short (min 6 chars).</div>';
                }
            } else {
                $profile_message = '<div class="alert alert-danger">Old password is incorrect.</div>';
            }
            $stmt->close();
        }
    } elseif ($_POST['action'] === 'submit_feedback') {
        $rating = isset($_POST['rating']) ? intval($_POST['rating']) : 0;
        $feedback = isset($_POST['feedback']) ? trim($_POST['feedback']) : '';
        
        if ($rating > 0 && !empty($feedback)) {
            $feedback_query = "INSERT INTO reviews (user_id, booking_id, rating, review_text, status) VALUES (?, NULL, ?, ?, 'pending')";
            $stmt = $conn->prepare($feedback_query);
            if ($stmt) {
                $stmt->bind_param('iis', $user_id, $rating, $feedback);
                if ($stmt->execute()) {
                    $profile_message = '<div class="alert alert-success">Feedback submitted successfully! Admin will review it.</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error submitting feedback.</div>';
                }
                $stmt->close();
            }
        } else {
            $profile_message = '<div class="alert alert-danger">Please fill all fields.</div>';
        }
    } elseif ($_POST['action'] === 'submit_contact') {
        $contact_subject = isset($_POST['contact_subject']) ? trim($_POST['contact_subject']) : '';
        $contact_message = isset($_POST['contact_message']) ? trim($_POST['contact_message']) : '';
        
        if (!empty($contact_subject) && !empty($contact_message)) {
            $contact_query = "INSERT INTO contact_messages (name, email, phone, subject, message, status) VALUES (?, ?, ?, ?, ?, 'new')";
            $stmt = $conn->prepare($contact_query);
            if ($stmt) {
                $user_email = $_SESSION['user_email'] ?? '';
                $user_phone = '';
                // Get user phone
                $phone_query = "SELECT phone FROM users WHERE id = ?";
                $phone_stmt = $conn->prepare($phone_query);
                if ($phone_stmt) {
                    $phone_stmt->bind_param('i', $user_id);
                    $phone_stmt->execute();
                    $phone_result = $phone_stmt->get_result();
                    if ($phone_row = $phone_result->fetch_assoc()) {
                        $user_phone = $phone_row['phone'];
                    }
                    $phone_stmt->close();
                }
                
                $stmt->bind_param('sssss', $user_name, $user_email, $user_phone, $contact_subject, $contact_message);
                if ($stmt->execute()) {
                    $profile_message = '<div class="alert alert-success">Contact message sent to admin! Admin will respond soon.</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error sending contact message.</div>';
                }
                $stmt->close();
            }
        }
    } elseif ($_POST['action'] === 'book_service') {
        $service_id = isset($_POST['service_id']) ? intval($_POST['service_id']) : 0;
        $booking_date = isset($_POST['booking_date']) ? $_POST['booking_date'] : '';
        $booking_time = isset($_POST['booking_time']) ? $_POST['booking_time'] : '';
        
        if ($service_id > 0 && !empty($booking_date) && !empty($booking_time)) {
            $booking_query = "INSERT INTO bookings (user_id, service_id, booking_date, booking_time, status) VALUES (?, ?, ?, ?, 'pending')";
            $stmt = $conn->prepare($booking_query);
            if ($stmt) {
                $stmt->bind_param('iiss', $user_id, $service_id, $booking_date, $booking_time);
                if ($stmt->execute()) {
                    $profile_message = '<div class="alert alert-success">Service booked successfully! Admin will confirm soon.</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error booking service.</div>';
                }
                $stmt->close();
            }
        }
    }
}

// Fetch user profile data
$user_query = "SELECT name, email, phone FROM users WHERE id = ?";
$user_stmt = $conn->prepare($user_query);
$user_stmt->bind_param('i', $user_id);
$user_stmt->execute();
$user_result = $user_stmt->get_result();
$user_data = $user_result->fetch_assoc();
$user_stmt->close();

// Fetch services for booking
$services_query = "SELECT id, name, category, price FROM services WHERE status = 'active' ORDER BY name ASC";
$services_result = $conn->query($services_query);
$services = array();
if ($services_result && $services_result->num_rows > 0) {
    while ($row = $services_result->fetch_assoc()) {
        $services[] = $row;
    }
}

// Fetch user bookings
$bookings_query = "SELECT b.id, s.name as service_name, b.booking_date, b.booking_time, b.status FROM bookings b JOIN services s ON b.service_id = s.id WHERE b.user_id = ? ORDER BY b.created_at DESC";
$bookings_stmt = $conn->prepare($bookings_query);
$bookings_stmt->bind_param('i', $user_id);
$bookings_stmt->execute();
$bookings_result = $bookings_stmt->get_result();
$bookings = array();
if ($bookings_result && $bookings_result->num_rows > 0) {
    while ($row = $bookings_result->fetch_assoc()) {
        $bookings[] = $row;
    }
}
$bookings_stmt->close();

// Fetch user feedback
$feedback_query = "SELECT id, rating, review_text, status, created_at FROM reviews WHERE user_id = ? ORDER BY created_at DESC";
$feedback_stmt = $conn->prepare($feedback_query);
$feedback_stmt->bind_param('i', $user_id);
$feedback_stmt->execute();
$feedback_result = $feedback_stmt->get_result();
$user_feedback = array();
if ($feedback_result && $feedback_result->num_rows > 0) {
    while ($row = $feedback_result->fetch_assoc()) {
        $user_feedback[] = $row;
    }
}
$feedback_stmt->close();

// Fetch contact messages
$contact_query = "SELECT id, subject, message, status, created_at FROM contact_messages WHERE name = ? OR email = ? ORDER BY created_at DESC";
$contact_stmt = $conn->prepare($contact_query);
$contact_stmt->bind_param('ss', $user_name, $user_data['email']);
$contact_stmt->execute();
$contact_result = $contact_stmt->get_result();
$contact_messages = array();
if ($contact_result && $contact_result->num_rows > 0) {
    while ($row = $contact_result->fetch_assoc()) {
        $contact_messages[] = $row;
    }
}
$contact_stmt->close();
?><!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Customer Dashboard | Cleaning Service</title>
    <link rel="stylesheet" href="assets/css/bootstrap.min.css">
    <link rel="stylesheet" href="assets/css/fontawesome.min.css">
    <link rel="shortcut icon" href="assets/img/favicon.ico" type="image/x-icon">
    <link rel="icon" href="assets/img/favicon.ico" type="image/x-icon">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin="">
    <link href="../../css2-2?family=Fuzzy+Bubbles:wght@400;700&family=Poppins:wght@400;500;600;700;800&family=Rubik:ital,wght@0,300..900;1,300..900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/bootstrap.min.css">
    <link rel="stylesheet" href="assets/css/fontawesome.min.css">
    <link rel="stylesheet" href="assets/css/magnific-popup.min.css">
    <link rel="stylesheet" href="assets/css/slick.min.css">
    <link rel="stylesheet" href="assets/css/animate.min.css">
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        :root {
            --primary: #109c3d;
            --dark: #0f8c3d;
            --light: #f8fbff;
            --border: #e2e8f0;
        }

        * { transition: all 0.3s ease; }

        body {
            background: linear-gradient(135deg, #f8fbff, #f1f5f4);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .dashboard-wrapper {
            display: grid;
            grid-template-columns: 260px 1fr;
            min-height: 100vh;
        }

        .sidebar {
            background: linear-gradient(135deg, var(--primary), var(--dark));
            color: white;
            padding: 30px 20px;
            position: sticky;
            top: 0;
            height: auto;
            overflow-y: auto;
        }

        .sidebar-brand {
            font-weight: 700;
            font-size: 18px;
            margin-bottom: 30px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .sidebar-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .sidebar-menu a {
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            padding: 12px 15px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            gap: 12px;
            cursor: pointer;
            margin-bottom: 8px;
        }

        .sidebar-menu a:hover,
        .sidebar-menu a.active {
            background: rgba(255, 255, 255, 0.2);
            color: white;
        }

        .logout-btn {
            margin-top: 30px;
            padding-top: 30px;
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        .main-content {
            padding: 40px;
            overflow-y: auto;
        }

        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 40px;
            flex-wrap: wrap;
            gap: 20px;
        }

        .header h1 {
            font-size: 32px;
            font-weight: 700;
            color: #0f172a;
        }

        .avatar-box {
            display: flex;
            align-items: center;
            gap: 15px;
            background: white;
            padding: 15px 20px;
            border-radius: 12px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        .avatar {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 700;
        }

        .content-section {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
            margin-bottom: 30px;
            display: none;
        }

        .content-section.active {
            display: block;
        }

        .section-title {
            font-size: 22px;
            font-weight: 700;
            color: #0f172a;
            margin-bottom: 25px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            font-weight: 600;
            color: #0f172a;
            margin-bottom: 8px;
            display: block;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 14px;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(16, 156, 61, 0.1);
        }

        .btn-submit {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
        }

        .btn-submit:hover {
            background: var(--dark);
        }

        .booking-item,
        .feedback-item,
        .contact-item {
            background: #f8fafc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 15px;
            border-left: 4px solid var(--primary);
        }

        .status-badge {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-top: 10px;
        }

        .status-pending {
            background: rgba(251, 191, 36, 0.1);
            color: #d97706;
        }

        .status-confirmed {
            background: rgba(59, 130, 246, 0.1);
            color: #2563eb;
        }

        .status-completed {
            background: rgba(16, 156, 61, 0.1);
            color: var(--primary);
        }

        .status-approved {
            background: rgba(16, 156, 61, 0.1);
            color: var(--primary);
        }

        .status-rejected,
        .status-new {
            background: rgba(239, 68, 68, 0.1);
            color: #ef4444;
        }

        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: rgba(16, 156, 61, 0.1);
            color: var(--primary);
            border: 1px solid var(--primary);
        }

        .alert-danger {
            background: rgba(239, 68, 68, 0.1);
            color: #ef4444;
            border: 1px solid #ef4444;
        }

        .rating-stars {
            color: #fbbf24;
            font-size: 18px;
            letter-spacing: 5px;
        }

        .rating-select {
            display: flex;
            gap: 8px;
            margin-top: 10px;
        }

        .rating-select button {
            width: 40px;
            height: 40px;
            border: 2px solid var(--border);
            background: white;
            border-radius: 8px;
            cursor: pointer;
            font-size: 18px;
        }

        .rating-select button:hover,
        .rating-select button.active {
            border-color: var(--primary);
            background: var(--primary);
            color: white;
        }

        /* Overview Stats Cards */
        .overview-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: linear-gradient(135deg, #f8fbff, #f1f5f4);
            padding: 25px;
            border-radius: 12px;
            border-left: 5px solid var(--primary);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.12);
        }

        .stat-card.pending { border-left-color: #fbbf24; }
        .stat-card.confirmed { border-left-color: #3b82f6; }
        .stat-card.completed { border-left-color: #109c3d; }
        .stat-card.total { border-left-color: #8b5cf6; }

        .stat-card-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
        }

        .stat-card-title {
            font-size: 14px;
            color: #64748b;
            font-weight: 500;
        }

        .stat-card-icon {
            font-size: 24px;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 10px;
            background: rgba(16, 156, 61, 0.1);
        }

        .stat-card.pending .stat-card-icon { background: rgba(251, 191, 36, 0.1); color: #fbbf24; }
        .stat-card.confirmed .stat-card-icon { background: rgba(59, 130, 246, 0.1); color: #3b82f6; }
        .stat-card.completed .stat-card-icon { background: rgba(16, 156, 61, 0.1); color: #109c3d; }
        .stat-card.total .stat-card-icon { background: rgba(139, 92, 246, 0.1); color: #8b5cf6; }

        .stat-card-value {
            font-size: 32px;
            font-weight: 700;
            color: #0f172a;
            margin-bottom: 5px;
        }

        .stat-card-subtitle {
            font-size: 12px;
            color: #94a3b8;
        }

        .recent-activity {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            margin-bottom: 30px;
        }

        .recent-activity-title {
            font-size: 18px;
            font-weight: 700;
            color: #0f172a;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .activity-item {
            display: flex;
            align-items: center;
            gap: 15px;
            padding: 12px 0;
            border-bottom: 1px solid #f0f0f0;
        }

        .activity-item:last-child {
            border-bottom: none;
        }

        .activity-icon {
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            background: rgba(16, 156, 61, 0.1);
            color: var(--primary);
            font-size: 16px;
            flex-shrink: 0;
        }

        .activity-content {
            flex: 1;
        }

        .activity-title {
            font-size: 14px;
            font-weight: 600;
            color: #0f172a;
            margin-bottom: 3px;
        }

        .activity-date {
            font-size: 12px;
            color: #94a3b8;
        }

        .activity-status {
            font-size: 12px;
            font-weight: 600;
            padding: 4px 10px;
            border-radius: 6px;
            background: rgba(16, 156, 61, 0.1);
            color: var(--primary);
        }

        @media (max-width: 768px) {
            .dashboard-wrapper {
                grid-template-columns: 1fr;
            }

            .sidebar {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                width: 260px;
                height: 100vh;
                z-index: 1000;
                background: #063a41;
                overflow-y: auto;
            }

            .sidebar.active {
                display: block;
            }

            .main-content {
                padding: 20px;
            }

            .header {
                display: flex;
                align-items: center;
                gap: 15px;
            }

            .sidebar-toggle {
                display: inline-block !important;
                background: none;
                border: none;
                cursor: pointer;
                font-size: 24px;
                color: #109c3d;
                padding: 0;
                margin: 0;
            }

            .header h1 {
                font-size: 24px;
                margin: 0;
                flex: 1;
            }
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    <div class="dashboard-wrapper">
        <!-- Sidebar -->
        <aside class="sidebar" id="sidebar">
            <div class="sidebar-brand">
                <i class="fas fa-leaf"></i> Dashboard
            </div>
            <ul class="sidebar-menu">
                <li><a class="menu-link active" data-tab="overview"><i class="fas fa-chart-line"></i> Overview</a></li>
                <li><a class="menu-link" data-tab="booking"><i class="fas fa-shopping-cart"></i> Service Booking</a></li>
                <li><a class="menu-link" data-tab="profile"><i class="fas fa-user"></i> Profile</a></li>
                <li><a class="menu-link" data-tab="feedback"><i class="fas fa-comment"></i> Feedback</a></li>
                <li><a class="menu-link" data-tab="contact"><i class="fas fa-envelope"></i> Contact</a></li>
            </ul>
            <div class="logout-btn">
                <a href="logout.php" style="color: rgba(255,255,255,0.8); text-decoration: none;"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <div class="header">
                <button class="sidebar-toggle" id="sidebarToggle" style="display: none; background: none; border: none; cursor: pointer; font-size: 24px; color: #109c3d;">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Welcome, <?php echo htmlspecialchars($user_name); ?>!</h1>
                <div class="avatar-box">
                    <div class="avatar"><?php echo strtoupper(substr($user_name, 0, 1)); ?></div>
                    <div>
                        <strong><?php echo htmlspecialchars($user_name); ?></strong>
                        <p style="margin: 0; font-size: 12px; color: #64748b;">Customer</p>
                    </div>
                </div>
            </div>

            <?php echo $profile_message; ?>

            <!-- OVERVIEW SECTION -->
            <div id="overview" class="content-section active">
                <h2 class="section-title"><i class="fas fa-chart-line"></i> Dashboard Overview</h2>

                <!-- Stats Cards -->
                <div class="overview-grid">
                    <div class="stat-card total">
                        <div class="stat-card-header">
                            <div class="stat-card-title">Total Bookings</div>
                            <div class="stat-card-icon"><i class="fas fa-list"></i></div>
                        </div>
                        <div class="stat-card-value"><?php echo count($bookings); ?></div>
                        <div class="stat-card-subtitle">All time services booked</div>
                    </div>

                    <div class="stat-card pending">
                        <div class="stat-card-header">
                            <div class="stat-card-title">Pending</div>
                            <div class="stat-card-icon"><i class="fas fa-clock"></i></div>
                        </div>
                        <div class="stat-card-value">
                            <?php 
                                $pending_count = 0;
                                foreach ($bookings as $b) {
                                    if ($b['status'] === 'pending') $pending_count++;
                                }
                                echo $pending_count;
                            ?>
                        </div>
                        <div class="stat-card-subtitle">Awaiting approval</div>
                    </div>

                    <div class="stat-card confirmed">
                        <div class="stat-card-header">
                            <div class="stat-card-title">Confirmed</div>
                            <div class="stat-card-icon"><i class="fas fa-check-circle"></i></div>
                        </div>
                        <div class="stat-card-value">
                            <?php 
                                $confirmed_count = 0;
                                foreach ($bookings as $b) {
                                    if ($b['status'] === 'confirmed') $confirmed_count++;
                                }
                                echo $confirmed_count;
                            ?>
                        </div>
                        <div class="stat-card-subtitle">Approved bookings</div>
                    </div>

                    <div class="stat-card completed">
                        <div class="stat-card-header">
                            <div class="stat-card-title">Completed</div>
                            <div class="stat-card-icon"><i class="fas fa-star"></i></div>
                        </div>
                        <div class="stat-card-value">
                            <?php 
                                $completed_count = 0;
                                foreach ($bookings as $b) {
                                    if ($b['status'] === 'completed') $completed_count++;
                                }
                                echo $completed_count;
                            ?>
                        </div>
                        <div class="stat-card-subtitle">Finished services</div>
                    </div>
                </div>

                <!-- Recent Activity -->
                <div class="recent-activity">
                    <div class="recent-activity-title">
                        <i class="fas fa-history"></i> Recent Activity
                    </div>
                    
                    <?php if (empty($bookings)): ?>
                        <div style="text-align: center; padding: 30px; color: #94a3b8;">
                            <i class="fas fa-inbox" style="font-size: 40px; margin-bottom: 10px; opacity: 0.5;"></i>
                            <p style="margin: 0;">No bookings yet. Start by booking a service!</p>
                        </div>
                    <?php else: ?>
                        <?php 
                            $limited_bookings = array_slice($bookings, 0, 5);
                            foreach ($limited_bookings as $booking): 
                        ?>
                            <div class="activity-item">
                                <div class="activity-icon">
                                    <i class="fas fa-shopping-bag"></i>
                                </div>
                                <div class="activity-content">
                                    <div class="activity-title"><?php echo htmlspecialchars($booking['service_name']); ?></div>
                                    <div class="activity-date">
                                        <?php echo date('M d, Y', strtotime($booking['booking_date'])) . ' at ' . date('h:i A', strtotime($booking['booking_time'])); ?>
                                    </div>
                                </div>
                                <div class="activity-status status-<?php echo $booking['status']; ?>">
                                    <?php echo ucfirst($booking['status']); ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Quick Stats -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 20px;">
                    <div style="background: white; padding: 20px; border-radius: 12px; box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);">
                        <div style="font-weight: 600; color: #0f172a; margin-bottom: 15px; display: flex; align-items: center; gap: 10px;">
                            <i class="fas fa-star" style="color: #fbbf24;"></i> Your Feedback
                        </div>
                        <div style="font-size: 28px; font-weight: 700; color: #0f172a; margin-bottom: 5px;">
                            <?php echo count($user_feedback); ?>
                        </div>
                        <div style="font-size: 13px; color: #94a3b8;">Feedback submissions</div>
                    </div>

                    <div style="background: white; padding: 20px; border-radius: 12px; box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);">
                        <div style="font-weight: 600; color: #0f172a; margin-bottom: 15px; display: flex; align-items: center; gap: 10px;">
                            <i class="fas fa-envelope" style="color: #3b82f6;"></i> Messages
                        </div>
                        <div style="font-size: 28px; font-weight: 700; color: #0f172a; margin-bottom: 5px;">
                            <?php echo count($contact_messages); ?>
                        </div>
                        <div style="font-size: 13px; color: #94a3b8;">Contact messages sent</div>
                    </div>

                    <div style="background: white; padding: 20px; border-radius: 12px; box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);">
                        <div style="font-weight: 600; color: #0f172a; margin-bottom: 15px; display: flex; align-items: center; gap: 10px;">
                            <i class="fas fa-user" style="color: #109c3d;"></i> Profile
                        </div>
                        <div style="font-size: 14px; color: #0f172a; margin-bottom: 5px;">
                            <?php echo htmlspecialchars($user_data['name']); ?>
                        </div>
                        <div style="font-size: 13px; color: #94a3b8;">Last updated: <?php echo date('M d, Y'); ?></div>
                    </div>
                </div>
            </div>

            <div id="booking" class="content-section">
                <h2 class="section-title"><i class="fas fa-shopping-cart"></i> Book a Service</h2>

                <form method="POST" action="">
                    <input type="hidden" name="action" value="book_service">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Select Service *</label>
                                <select name="service_id" required>
                                    <option value="">-- Choose a Service --</option>
                                    <?php foreach ($services as $svc): ?>
                                        <option value="<?php echo $svc['id']; ?>">
                                            <?php echo htmlspecialchars($svc['name']); ?> - $<?php echo $svc['price']; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Booking Date *</label>
                                <input type="date" name="booking_date" required min="<?php echo date('Y-m-d'); ?>">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Booking Time *</label>
                                <input type="time" name="booking_time" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group" style="padding-top: 33px;">
                                <button type="submit" class="btn-submit w-100">Book Service</button>
                            </div>
                        </div>
                    </div>
                </form>

                <h3 style="margin-top: 40px; margin-bottom: 20px;">Your Bookings</h3>
                <?php if (count($bookings) > 0): ?>
                    <?php foreach ($bookings as $booking): ?>
                        <div class="booking-item">
                            <h4><?php echo htmlspecialchars($booking['service_name']); ?></h4>
                            <p><strong>Date:</strong> <?php echo date('F d, Y', strtotime($booking['booking_date'])); ?></p>
                            <p><strong>Time:</strong> <?php echo $booking['booking_time']; ?></p>
                            <span class="status-badge status-<?php echo $booking['status']; ?>">
                                <?php echo ucfirst($booking['status']); ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <p style="text-align: center; color: #64748b;">No bookings yet. Book your first service!</p>
                <?php endif; ?>
            </div>

            <!-- PROFILE -->
            <div id="profile" class="content-section">
                <h2 class="section-title"><i class="fas fa-user"></i> Edit Profile</h2>

                <form method="POST" action="">
                    <input type="hidden" name="action" value="update_profile">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Full Name</label>
                                <input type="text" name="name" value="<?php echo htmlspecialchars($user_data['name'] ?? ''); ?>" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Email</label>
                                <input type="email" value="<?php echo htmlspecialchars($user_data['email'] ?? ''); ?>" disabled>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Phone Number</label>
                                <input type="tel" name="phone" value="<?php echo htmlspecialchars($user_data['phone'] ?? ''); ?>" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group" style="padding-top: 33px;">
                                <button type="submit" class="btn-submit w-100">Update Profile</button>
                            </div>
                        </div>
                    </div>
                </form>

                <h3 style="margin-top: 40px; margin-bottom: 20px;">Change Password</h3>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="update_password">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Old Password</label>
                                <input type="password" name="old_password" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>New Password</label>
                                <input type="password" name="new_password" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Confirm Password</label>
                                <input type="password" name="confirm_password" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group" style="padding-top: 33px;">
                                <button type="submit" class="btn-submit w-100">Change Password</button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>

            <!-- FEEDBACK -->
            <div id="feedback" class="content-section">
                <h2 class="section-title"><i class="fas fa-comment"></i> Leave Feedback</h2>

                <form method="POST" action="">
                    <input type="hidden" name="action" value="submit_feedback">
                    <div class="form-group">
                        <label>Rating (1-5 stars)</label>
                        <div class="rating-select">
                            <button type="button" class="star-btn" data-rating="1">★</button>
                            <button type="button" class="star-btn" data-rating="2">★</button>
                            <button type="button" class="star-btn" data-rating="3">★</button>
                            <button type="button" class="star-btn" data-rating="4">★</button>
                            <button type="button" class="star-btn" data-rating="5">★</button>
                        </div>
                        <input type="hidden" name="rating" id="rating-input" value="0">
                    </div>
                    <div class="form-group">
                        <label>Your Feedback</label>
                        <textarea name="feedback" rows="5" placeholder="Share your experience..." required></textarea>
                    </div>
                    <button type="submit" class="btn-submit">Submit Feedback</button>
                </form>

                <h3 style="margin-top: 40px; margin-bottom: 20px;">Your Feedback</h3>
                <?php if (count($user_feedback) > 0): ?>
                    <?php foreach ($user_feedback as $fb): ?>
                        <div class="feedback-item">
                            <div class="rating-stars">
                                <?php for ($i = 0; $i < $fb['rating']; $i++): ?>★<?php endfor; ?>
                            </div>
                            <p><?php echo htmlspecialchars($fb['review_text']); ?></p>
                            <small style="color: #64748b;">Submitted on <?php echo date('F d, Y', strtotime($fb['created_at'])); ?></small>
                            <span class="status-badge status-<?php echo $fb['status']; ?>">
                                <?php echo ucfirst($fb['status']); ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <p style="text-align: center; color: #64748b;">No feedback submitted yet.</p>
                <?php endif; ?>
            </div>

            <!-- CONTACT -->
            <div id="contact" class="content-section">
                <h2 class="section-title"><i class="fas fa-envelope"></i> Contact Admin</h2>

                <form method="POST" action="">
                    <input type="hidden" name="action" value="submit_contact">
                    <div class="form-group">
                        <label>Subject</label>
                        <input type="text" name="contact_subject" placeholder="What is this about?" required>
                    </div>
                    <div class="form-group">
                        <label>Message</label>
                        <textarea name="contact_message" rows="5" placeholder="Type your message here..." required></textarea>
                    </div>
                    <button type="submit" class="btn-submit">Send Message</button>
                </form>

                <h3 style="margin-top: 40px; margin-bottom: 20px;">Contact History</h3>
                <?php if (count($contact_messages) > 0): ?>
                    <?php foreach ($contact_messages as $msg): ?>
                        <div class="contact-item">
                            <h4><?php echo htmlspecialchars($msg['subject']); ?></h4>
                            <p><?php echo htmlspecialchars($msg['message']); ?></p>
                            <small style="color: #64748b;">Sent on <?php echo date('F d, Y H:i', strtotime($msg['created_at'])); ?></small>
                            <span class="status-badge status-<?php echo $msg['status']; ?>">
                                <?php echo ucfirst($msg['status']); ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <p style="text-align: center; color: #64748b;">No contact messages yet.</p>
                <?php endif; ?>
            </div>
        </main>
    </div>
    <script src="assets/js/vendor/jquery-3.7.1.min.js"></script>
    <script src="assets/js/slick.min.js"></script>
    <script src="assets/js/bootstrap.min.js"></script>
    <script src="assets/js/jquery.magnific-popup.min.js"></script>
    <script src="assets/js/imagesloaded.pkgd.min.js"></script>
    <script src="assets/js/gsap.min.js"></script>
    <script src="assets/js/ScrollTrigger.min.js"></script>
    <script src="assets/js/gsap-scroll-to-plugin.js"></script>
    <script src="assets/js/SplitText.js"></script>
    <script src="assets/js/lenis.min.js"></script>
    <script src="assets/js/wow.min.js"></script>
    <script src="assets/js/main.js"></script>
    <script src="assets/js/vendor/jquery-3.7.1.min.js"></script>
    <script>
        // Tab switching
        document.querySelectorAll('.menu-link').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                
                document.querySelectorAll('.menu-link').forEach(l => l.classList.remove('active'));
                this.classList.add('active');
                
                document.querySelectorAll('.content-section').forEach(s => s.classList.remove('active'));
                const tabName = this.getAttribute('data-tab');
                document.getElementById(tabName).classList.add('active');
                
                // Close sidebar on mobile when menu item is clicked
                const sidebar = document.getElementById('sidebar');
                if (sidebar && window.innerWidth <= 768) {
                    sidebar.classList.remove('active');
                }
            });
        });

        // Rating stars
        document.querySelectorAll('.star-btn').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                const rating = this.getAttribute('data-rating');
                document.getElementById('rating-input').value = rating;
                
                document.querySelectorAll('.star-btn').forEach(b => b.classList.remove('active'));
                let index = Array.from(document.querySelectorAll('.star-btn')).indexOf(this);
                document.querySelectorAll('.star-btn').forEach((b, i) => {
                    if (i <= index) b.classList.add('active');
                });
            });
        });

        // Mobile Sidebar Toggle
        function handleSidebarToggle() {
            const sidebar = document.getElementById('sidebar');
            const toggle = document.getElementById('sidebarToggle');
            const isMobile = window.innerWidth <= 768;
            
            if (isMobile) {
                toggle.style.display = 'inline-block';
                sidebar.classList.remove('active');
            } else {
                toggle.style.display = 'none';
                sidebar.classList.remove('active');
            }
        }

        // Initialize on load
        document.addEventListener('DOMContentLoaded', function() {
            handleSidebarToggle();
            
            const toggle = document.getElementById('sidebarToggle');
            const sidebar = document.getElementById('sidebar');
            
            // Toggle sidebar on button click
            toggle.addEventListener('click', function() {
                sidebar.classList.toggle('active');
            });
            
            // Close sidebar when clicking outside
            document.addEventListener('click', function(e) {
                if (!e.target.closest('#sidebar') && !e.target.closest('#sidebarToggle')) {
                    sidebar.classList.remove('active');
                }
            });
        });

        // Handle window resize
        window.addEventListener('resize', handleSidebarToggle);
    </script>
</body>
</html>

<?php include 'footer.php'; ?>